#include <QDebug>

#include <stdio.h>
#include <stdint.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <string.h>

#include <linux/spi/spidev.h>

#include "spidev.h"

SpiDev::SpiDev(const char * devName){
    _devName = NULL;
    if (devName != NULL ){
        _devName = (char *)malloc(strlen(devName)+1);
        if (_devName != NULL)
            strcpy(_devName,devName);
    }
    _isOpen = false;
}

SpiDev::SpiDev(const char * devName, spidev_cfg_t * cfg){
    _devName = NULL;
    if (devName != NULL ){
        _devName = (char *)malloc(strlen(devName)+1);
        if (_devName != NULL)
            strcpy(_devName, devName);
    }
    if (cfg != NULL){
        memcpy(&_cfg, cfg, sizeof(spidev_cfg_t));
    }
    else {
        _cfg.mode = 0;
        _cfg.speed = 1000000;
        _cfg.bits_per_word = 8;
        _cfg.delay = 0;

    }
    _isOpen = false;
}

SpiDev::~SpiDev(){
    qDebug() << "SpiDev destructor";
    if (_devName != nullptr ) {
        free(_devName);
        _devName = nullptr;
    }
    if (_isOpen)
        close(_fd);
}

bool SpiDev::setBitPerWord(uint8_t nbBits){

    /* Set bits per word*/
    if (ioctl(_fd, SPI_IOC_WR_BITS_PER_WORD, &nbBits) < 0) {
        return false;
    }
    if (ioctl(_fd, SPI_IOC_RD_BITS_PER_WORD, &nbBits) < 0) {
        return false;
    }

    _cfg.bits_per_word = nbBits;

    return true;
}

bool SpiDev::setSpeed(uint32_t speed){
    /* Set SPI speed*/
    if (ioctl(_fd, SPI_IOC_WR_MAX_SPEED_HZ, &speed) < 0) {
        return false;
    }
    if (ioctl(_fd, SPI_IOC_RD_MAX_SPEED_HZ, &speed) < 0) {
        return false;
    }

    _cfg.speed = speed;

    return true;
}

bool SpiDev::setMode(uint8_t mode){
    /* Set SPI_POL and SPI_PHA */
    if (ioctl(_fd, SPI_IOC_WR_MODE, &mode) < 0) {
        return false;
    }
    if (ioctl(_fd, SPI_IOC_RD_MODE, &mode) < 0) {
        return false;
    }

    _cfg.mode = mode;

    return true;
}

int SpiDev::xfer(uint8_t *txBuf, uint8_t txLen, uint8_t *rxBuf, uint8_t rxLen){
    struct spi_ioc_transfer spi_message[1];

    memset(spi_message, 0, sizeof(spi_message));

    spi_message[0].rx_buf = (unsigned long)rxBuf;
    spi_message[0].tx_buf = (unsigned long)txBuf;
    spi_message[0].len = txLen;
    return ioctl(_fd, SPI_IOC_MESSAGE(1), spi_message);
}

int SpiDev::write(uint8_t *txBuf,uint8_t txLen){
    struct spi_ioc_transfer spi_message[1];

    memset(spi_message, 0, sizeof(spi_message));

    spi_message[0].tx_buf = (unsigned long)txBuf;
    spi_message[0].len = txLen;

    return ioctl(_fd, SPI_IOC_MESSAGE(1), spi_message);
}

int SpiDev::read(uint8_t *rxBuf,uint8_t rxLen){
    struct spi_ioc_transfer spi_message[1];

    memset(spi_message, 0, sizeof(spi_message));

    spi_message[0].rx_buf = (unsigned long)rxBuf;
    spi_message[0].len = rxLen;

    return ioctl(_fd, SPI_IOC_MESSAGE(1), spi_message);
}

bool SpiDev::begin(){
    /* open spidev device */
    if (_isOpen == true )
        return true;
    if (_devName == NULL)
        return false;
    _fd = open(_devName, O_RDWR);

    if (_fd < 0) {
        return false;
    }
    /* Set SPI_POL and SPI_PHA */
    if (ioctl(_fd, SPI_IOC_WR_MODE, &_cfg.mode) < 0) {
        close(_fd);
        return false;
    }
    if (ioctl(_fd, SPI_IOC_RD_MODE, &_cfg.mode) < 0) {
        close(_fd);
        return false;
    }

    /* Set bits per word*/
    if (ioctl(_fd, SPI_IOC_WR_BITS_PER_WORD, &_cfg.bits_per_word) < 0) {
        close(_fd);
        return false;
    }
    if (ioctl(_fd, SPI_IOC_RD_BITS_PER_WORD, &_cfg.bits_per_word) < 0) {
        close(_fd);
        return false;
    }

    /* Set SPI speed*/
    if (ioctl(_fd, SPI_IOC_WR_MAX_SPEED_HZ, &_cfg.speed) < 0) {
        close(_fd);
        return false;
    }
    if (ioctl(_fd, SPI_IOC_RD_MAX_SPEED_HZ, &_cfg.speed) < 0) {
        close(_fd);
        return false;
    }

    _isOpen = true;

    return true;
}

bool SpiDev::setConfig(spidev_cfg_t *cfg){
    if (cfg != NULL){
        memcpy(&_cfg,cfg,sizeof(spidev_cfg_t));
        if (_isOpen){
            /* Set SPI_POL and SPI_PHA */
            if (ioctl(_fd, SPI_IOC_WR_MODE, &_cfg.mode) < 0) {
                close(_fd);
                return false;
            }
            if (ioctl(_fd, SPI_IOC_RD_MODE, &_cfg.mode) < 0) {
                close(_fd);
                return false;
            }

            /* Set bits per word*/
            if (ioctl(_fd, SPI_IOC_WR_BITS_PER_WORD, &_cfg.bits_per_word) < 0) {
                close(_fd);
                return false;
            }
            if (ioctl(_fd, SPI_IOC_RD_BITS_PER_WORD, &_cfg.bits_per_word) < 0) {
                close(_fd);
                return false;
            }

            /* Set SPI speed*/
            if (ioctl(_fd, SPI_IOC_WR_MAX_SPEED_HZ, &_cfg.speed) < 0) {
                close(_fd);
                return false;
            }
            if (ioctl(_fd, SPI_IOC_RD_MAX_SPEED_HZ, &_cfg.speed) < 0) {
                close(_fd);
                return false;
            }
            return true;
        }
    }
    return false;
}



