// === appworker.cpp ===
#include <QThread>
#include <QVector>
#include <QFile>
#include <QDebug>

#include "appworker.h"

AppWorker::AppWorker(QObject *parent) : QObject{parent}
{
    _suspend = true;
    _quit = false;

    spidev_cfg_t spiCfg = {1, 8, 100000, 0};
    _dac8551 = new SpiDev("/dev/spidev0.0", &spiCfg);
}

AppWorker::~AppWorker()
{
    qDebug() << "AppWorker destructor";
    if (_dac8551) delete _dac8551;
}

void AppWorker::suspend()
{
    QMutexLocker locker(&_mutex);
    _suspend = true;
}

void AppWorker::resume()
{
    QMutexLocker locker(&_mutex);
    _suspend = false;
    _waitCondition.wakeAll();
}

void AppWorker::quit()
{
    QMutexLocker locker(&_mutex);
    _quit = true;
    _suspend = false;
    _waitCondition.wakeAll();
}

void AppWorker::setDataFile(QString fileName)
{
    _data.clear();
    QFile file(fileName);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        qWarning() << "Erreur ouverture fichier:" << fileName;
        return;
    }

    QTextStream stream(&file);
    int lineNumber = 0;
    while (!stream.atEnd()) {
        QString line = stream.readLine().trimmed();
        ++lineNumber;

        if (line.isEmpty()) continue;

        bool ok;
        int value = line.toInt(&ok);
        if (ok) {
            _data.append(value);
        } else {
            qWarning() << "Ligne invalide (" << lineNumber << "):" << line;
            _data.append(-1);
        }
    }
    file.close();
    qDebug() << "Fichier chargé avec" << _data.size() << "valeurs.";
}

void AppWorker::doWork()
{
    uint8_t spiBuffer[3];
    int idx = 0;

    if (_dac8551->begin()) {
        for (;;) {
            QMutexLocker locker(&_mutex);

            if (_quit) break;

            while (_suspend) _waitCondition.wait(&_mutex);

            if (_data.isEmpty()) {
                qWarning() << "Aucune donnée chargée, attente...";
                QThread::msleep(100);
                continue;
            }

            spiBuffer[0] = 0x00;
            spiBuffer[1] = (uint8_t)((_data[idx] >> 8) & 0x00ff);
            spiBuffer[2] = (uint8_t)(_data[idx] & 0x00ff);

            _dac8551->write(spiBuffer, 3);
            QThread::msleep(2);

            idx = (idx + 1) % _data.size();
        }
        emit finished();
    } else {
        qWarning() << "Erreur initialisation DAC8551";
    }
}
