// === app.cpp ===
#include <QCoreApplication>
#include <QDebug>
#include <QFileDialog>
#include <QMessageBox>


#include "app.h"

App::App(QObject *parent)
    : QObject{parent}
{
    _ihm = new MainUI();

    connect(_ihm, &MainUI::simuToggled, this, &App::onSimuToggled);
    connect(_ihm, &MainUI::exitRequested, this, &App::onExitRequested);
    connect(_ihm, &MainUI::openFileRequested, this, &App::onOpenFileClicked);

    _worker = new AppWorker();
    _workerThread = new QThread();
    _worker->moveToThread(_workerThread);

    connect(_workerThread, &QThread::started, _worker, &AppWorker::doWork);
    connect(_workerThread, &QThread::finished, [this]() {
        qDebug() << "Thread terminé";
        _workerThread->quit();
    });
    connect(_worker, &AppWorker::finished, [this]() {
        qDebug() << "Worker terminé";
        _workerThread->quit();
    });
    connect(_worker, &AppWorker::finished, _worker, &AppWorker::deleteLater);
    connect(_worker, &AppWorker::resultReady, [](const QString &result) {
        qDebug() << "Résultat reçu:" << result;
    });
}

App::~App()
{
    qDebug() << "App destructor";
}

void App::run()
{
    _ihm->show();
    if (!_selectedFile.isEmpty()) {
        _worker->setDataFile(_selectedFile);
    }
    _workerThread->start();
}

void App::onSimuToggled(bool checked)
{
    if (checked) {
        if (_selectedFile.isEmpty()) {
            QMessageBox::warning(_ihm, "Erreur", "Aucun fichier sélectionné.");
            _ihm->setFileNameLabel("Aucun fichier");
            return;
        }
        _worker->setDataFile(_selectedFile);
        _worker->resume();
    } else {
        _worker->suspend();
    }
}

void App::onExitRequested()
{
    _worker->quit();
    QCoreApplication::quit();
}

void App::onOpenFileClicked()
{
    QString fileName = QFileDialog::getOpenFileName(
        _ihm,
        tr("Ouvrir un fichier CSV"),
        "/home/pi/devQt/ecg/qt/app-simulateur/ECG signals (1000 fragments)",
        tr("Fichiers CSV (*.csv)")
    );

    if (!fileName.isEmpty()) {
        QFile file(fileName);
        if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
            QMessageBox::warning(_ihm, "Erreur", "Impossible d'ouvrir le fichier.");
            _ihm->setFileNameLabel("Aucun fichier sélectionné.");
            return;
        }
        if (file.size() == 0) {
            QMessageBox::warning(_ihm, "Erreur", "Le fichier est vide.");
            _ihm->setFileNameLabel("Aucun fichier sélectionné.");
            return;
        }

        file.close();
        _selectedFile = fileName;
        _ihm->setFileNameLabel(QFileInfo(fileName).fileName());

        QLineSeries *series =new QLineSeries;
        int x = 0;
        if (file.open(QIODevice::ReadOnly | QIODevice::Text)) {
            QTextStream in(&file);
            while (!in.atEnd()){
                QString line =in.readLine().trimmed();
                bool ok;
                int value = line.toInt(&ok);
                if (ok) {
                    series->append(x++, value);
                }
            }
            file.close();
        }

        QChart *chart =new QChart();
        chart->legend()->hide();
        chart->addSeries(series);
        chart->createDefaultAxes();
        _ihm->getChartView()->setChart(chart);
    }
}
